import { NextResponse } from "next/server";
import type { NextRequest } from "next/server";
import { getToken } from "next-auth/jwt";

export async function middleware(req: NextRequest) {
  const { pathname } = req.nextUrl;

  // allowlist: auth routes & static
  const allow = ["/api/auth", "/_next", "/favicon", "/assets"];
  if (allow.some(p => pathname === p || pathname.startsWith(p + "/"))) {
    return NextResponse.next();
  }

  const token = await getToken({ req, secret: process.env.NEXTAUTH_SECRET });

  // 👉 kalau SUDAH login & lagi buka /signin → redirect ke dashboard
  if ((pathname === "/signin" || pathname.startsWith("/signin")) && token) {
    return NextResponse.redirect(new URL("/dashboard", req.url));
  }

  // 👉 kalau BELUM login & buka path lain → lempar ke /signin
  if (!token && pathname !== "/signin") {
    const url = new URL("/signin", req.url);
    url.searchParams.set("callbackUrl", req.url);
    return NextResponse.redirect(url);
  }

  // role gate utk API destructive
  if (pathname.startsWith("/api/links")) {
    const method = req.method.toUpperCase();
    const role = (token as any)?.role || (token as any)?.user?.role;
    if ((method === "DELETE" || method === "PATCH") && role !== "ADMIN") {
      return NextResponse.json({ error: "Forbidden: admin only" }, { status: 403 });
    }
  }

  // Protect /admin routes
  if (pathname.startsWith("/admin") || pathname.startsWith("/api/users")) {
    const role = (token as any)?.role || (token as any)?.user?.role;
    if (role !== "ADMIN") {
      return NextResponse.redirect(new URL("/dashboard", req.url));
    }
  }

  return NextResponse.next();
}
export const config = { matcher: ["/:path*"] };
