import type { NextAuthOptions } from "next-auth";
import Credentials from "next-auth/providers/credentials";
import { PrismaAdapter } from "@next-auth/prisma-adapter";
import { PrismaClient } from "@prisma/client";
import bcrypt from "bcryptjs";
import { getServerSession } from "next-auth";

const prisma = new PrismaClient();

// typing ekstra (opsional)
declare module "next-auth" {
  interface Session {
    user?: { id?: string; email?: string | null; name?: string | null; role?: string };
  }
}

export const authOptions: NextAuthOptions = {
  // >>>>>>> PENTING: pakai JWT, bukan database
  session: { strategy: "jwt" },

  // Boleh tetap pakai adapter biar user tersimpan di DB
  adapter: PrismaAdapter(prisma),

  providers: [
    Credentials({
      name: "Credentials",
      credentials: {
        email: { label: "Email", type: "text" },
        password: { label: "Password", type: "password" }
      },
      async authorize(creds) {
        const email = creds?.email as string;
        const pass = creds?.password as string;
        if (!email || !pass) return null;
        const user = await prisma.user.findUnique({ where: { email } });
        if (!user) return null;
        const ok = await bcrypt.compare(pass, user.passwordHash);
        if (!ok) return null;
        // kembalikan minimal id + role
        return { id: user.id, email: user.email!, name: user.name, role: user.role };
      }
    })
  ],

  callbacks: {
    // masukin role & id ke token JWT
    async jwt({ token, user }) {
      if (user) {
        token.id = (user as any).id;
        token.role = (user as any).role;
      }
      return token;
    },
    // teruskan ke session (buat dipakai di UI & middleware)
    async session({ session, token }) {
      if (session.user) {
        (session.user as any).id = token.id as string | undefined;
        (session.user as any).role = token.role as string | undefined;
      }
      return session;
    },
  },

  pages: { signIn: "/signin" }
};

export function authSession() {
  return getServerSession(authOptions);
}
