﻿"use client";

import Link from "next/link";
import { usePathname } from "next/navigation";
import { useSession } from "next-auth/react";
import { useEffect, useMemo, useState } from "react";

type ToolItem = { name: string; slug: string; path: string; icon?: string; cate?: string };

export default function Sidebar() {
  const pathname = usePathname();
  const { data: session } = useSession();
  const [q, setQ] = useState("");
  const [tools, setTools] = useState<ToolItem[]>([]);
  const [collapsed, setCollapsed] = useState(false);
  const [isSmall, setIsSmall] = useState(false);
  const [tip, setTip] = useState<{ text: string; x: number; y: number; visible: boolean }>({ text: "", x: 0, y: 0, visible: false });

  useEffect(() => {
    (async () => {
      try {
        const r = await fetch("/api/tools");
        const j = await r.json();
        if (Array.isArray(j)) setTools(j);
      } catch {}
    })();
    // init responsive collapse
    const mq = window.matchMedia("(max-width: 1024px)");
    const onChange = () => setIsSmall(mq.matches);
    onChange();
    mq.addEventListener?.("change", onChange);
    // restore desktop preference
    const saved = localStorage.getItem("sidebarCollapsed");
    if (saved === "1") setCollapsed(true);
    return () => mq.removeEventListener?.("change", onChange);
  }, []);

  const items = useMemo(() => {
    const s = q.trim().toLowerCase();
    if (!s) return tools;
    return tools.filter(
      (t) => t.name.toLowerCase().includes(s) || t.slug.includes(s) || t.cate?.toLowerCase().includes(s)
    );
  }, [q, tools]);

  const isActive = (href: string) => pathname === href || pathname.startsWith(href + "/");
  const effCollapsed = isSmall || collapsed;

  function showTip(text: string, e: React.MouseEvent<HTMLElement>) {
    const rect = (e.currentTarget as HTMLElement).getBoundingClientRect();
    setTip({ text, x: rect.right + 10, y: rect.top + rect.height / 2, visible: true });
  }
  function hideTip() { setTip((t) => ({ ...t, visible: false })); }

  return (
    <aside
      className={`${effCollapsed ? "w-16" : "w-72"} shrink-0 h-screen sticky top-0 p-4 flex flex-col border-r sidebar-aside`}
      style={{ borderColor: "var(--line)", background: "rgba(23, 18, 43, .8)", backdropFilter: "blur(6px)" }}
    >
      <div className="mb-4 flex items-start justify-between">
        <div className="flex-1 flex justify-center">
          <img
            src="https://res.cloudinary.com/dfpdnfza4/image/upload/v1761342843/seosports_phnwwy.png"
            alt="Bre Logo"
          className={`${effCollapsed ? "w-10" : "w-44"} h-auto logo-glow`}
          />
        </div>
        {!isSmall && (
          <button
            aria-label={effCollapsed ? "Open sidebar" : "Collapse sidebar"}
            onClick={() => {
              const next = !collapsed;
              setCollapsed(next);
              localStorage.setItem("sidebarCollapsed", next ? "1" : "0");
            }}
            className="ml-2 text-sm rounded-lg px-2 py-1 hover:bg-[#1c1536]"
            title={effCollapsed ? "Open" : "Collapse"}
          >
            <span className="material-symbols-outlined">{effCollapsed ? "menu_open" : "close"}</span>
          </button>
        )}
      </div>

      {!effCollapsed && (
        <div className="mb-3">
          <input
            value={q}
            onChange={(e) => setQ(e.target.value)}
            placeholder="Cari tools"
            className="input"
          />
        </div>
      )}

      {!effCollapsed && (
        <div className="mb-2 mt-2 text-[11px] uppercase tracking-wide px-1" style={{ color: "var(--muted)" }}>
          Menu
        </div>
      )}
      <nav className="space-y-1">
        <Link
          href="/dashboard"
          className={`flex items-center gap-2 rounded-xl px-3 py-2 text-sm ${
            isActive("/dashboard") ? "bg-indigo-600 text-white" : "hover:bg-[#1c1536]"
          } ${effCollapsed ? 'tooltip' : ''}`}
          data-tip={effCollapsed ? 'Dashboard' : undefined}
          onMouseEnter={(e)=> effCollapsed && showTip('Dashboard', e)}
          onMouseLeave={()=> effCollapsed && hideTip()}
        >
          <span className="material-symbols-outlined">dashboard</span>
          {!effCollapsed && <span>Dashboard</span>}
        </Link>
        <Link
          href="/shortlinks"
          className={`flex items-center gap-2 rounded-xl px-3 py-2 text-sm ${
            isActive("/shortlinks") ? "bg-indigo-600 text-white" : "hover:bg-[#1c1536]"
          } ${effCollapsed ? 'tooltip' : ''}`}
          data-tip={effCollapsed ? 'Shortlinks' : undefined}
          onMouseEnter={(e)=> effCollapsed && showTip('Shortlinks', e)}
          onMouseLeave={()=> effCollapsed && hideTip()}
        >
          <span className="material-symbols-outlined">link</span>
          {!effCollapsed && <span>Shortlinks</span>}
        </Link>
        {session?.user?.role === "ADMIN" && (
          <Link
            href="/admin/users"
            className={`flex items-center gap-2 rounded-xl px-3 py-2 text-sm ${
              isActive("/admin/users") ? "bg-indigo-600 text-white" : "hover:bg-[#1c1536]"
            } ${effCollapsed ? 'tooltip' : ''}`}
            data-tip={effCollapsed ? 'Users' : undefined}
            onMouseEnter={(e)=> effCollapsed && showTip('Users', e)}
            onMouseLeave={()=> effCollapsed && hideTip()}
          >
            <span className="material-symbols-outlined">group</span>
            {!effCollapsed && <span>Users</span>}
          </Link>
        )}
        {session?.user?.role === "ADMIN" && (
          <Link
            href="/admin/tools"
            className={`flex items-center gap-2 rounded-xl px-3 py-2 text-sm ${
              isActive("/admin/tools") ? "bg-indigo-600 text-white" : "hover:bg-[#1c1536]"
            } ${effCollapsed ? 'tooltip' : ''}`}
            data-tip={effCollapsed ? 'Manage Tools' : undefined}
            onMouseEnter={(e)=> effCollapsed && showTip('Manage Tools', e)}
            onMouseLeave={()=> effCollapsed && hideTip()}
          >
            <span className="material-symbols-outlined">extension</span>
            {!effCollapsed && <span>Manage Tools</span>}
          </Link>
        )}
      </nav>

      {!effCollapsed && (
        <div className="mb-2 mt-4 text-[11px] uppercase tracking-wide px-1" style={{ color: "var(--muted)" }}>
          Tools
        </div>
      )}
      <nav className="space-y-1 overflow-y-auto pr-1 sidebar-nav">
        {items.map((t) => {
          const active = pathname.startsWith(`/tools/${t.slug}`);
          return (
            <Link
              key={t.slug}
              href={`/tools/${t.slug}`}
              className={`flex items-center gap-2 rounded-xl px-3 py-2 text-sm ${
                active ? "bg-indigo-600 text-white" : "hover:bg-[#1c1536]"
              } ${effCollapsed ? 'tooltip' : ''}`}
              data-tip={effCollapsed ? t.name : undefined}
              onMouseEnter={(e)=> effCollapsed && showTip(t.name ?? 'Tool', e)}
              onMouseLeave={()=> effCollapsed && hideTip()}
            >
              <span className="material-symbols-outlined">build</span>
              {!effCollapsed && <span className="truncate">{t.name}</span>}
            </Link>
          );
        })}
        {items.length === 0 && !effCollapsed && (
          <div className="text-xs px-2" style={{ color: "var(--muted)" }}>
            No tools found
          </div>
        )}
      </nav>

      {!effCollapsed && (
        <div className="mt-auto pt-4 text-[11px]" style={{ color: "var(--muted)" }}>
          <p>Powered By : <b>SeoSports</b></p>
        </div>
      )}
      {effCollapsed && tip.visible && (
        <div className="tooltip-fly" style={{ top: tip.y, left: tip.x }}>{tip.text}</div>
      )}
    </aside>
  );
}
