﻿"use client";

import { useEffect, useMemo, useState } from "react";
import Modal from "@/components/Modal";
import { Copy, Pencil, Trash2, RefreshCw } from "lucide-react";

type LinkRow = {
  id: string;
  slug: string;
  target: string;
  clicks: number;
  oneTime: boolean;
  expireAt: string | null;
  createdAt: string;
  updatedAt: string;
};

export default function ShortlinkTable() {
  const [q, setQ] = useState("");
  const [page, setPage] = useState(1);
  const [size, setSize] = useState(20);
  const [loading, setLoading] = useState(false);
  const [data, setData] = useState<{ total: number; items: LinkRow[] } | null>(null);
  const [copying, setCopying] = useState<string | null>(null);
  const [editOpen, setEditOpen] = useState(false);
  const [editing, setEditing] = useState<LinkRow | null>(null);
  const [saving, setSaving] = useState(false);
  const [deleting, setDeleting] = useState<string | null>(null);
  const [err, setErr] = useState<string | null>(null);
  const [role, setRole] = useState<string | null>(null);

  async function fetchData() {
    setLoading(true);
    setErr(null);
    try {
      const r = await fetch(`/api/links?q=${encodeURIComponent(q)}&page=${page}&size=${size}`);
      const j = await r.json();
      setData({ total: j.total, items: j.items });
    } catch (e: any) {
      setErr(e?.message || "Failed to load");
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => { fetchData(); }, [q, page, size]);
  useEffect(() => { (async () => { try { const r = await fetch('/api/whoami'); const j = await r.json(); setRole(j.user?.role || null); } catch {} })(); }, []);

  async function copyShort(slug: string) {
    try {
      const base = process.env.NEXT_PUBLIC_BASE_URL || window.location.origin;
      const url = `${base.replace(/\/$/, "")}/r/${slug}`;
      await navigator.clipboard.writeText(url);
      setCopying(slug);
      setTimeout(() => setCopying(null), 1200);
    } catch {}
  }

  function openEdit(row: LinkRow) {
    setEditing(row);
    setEditOpen(true);
  }

  async function saveEdit() {
    if (!editing) return;
    setSaving(true);
    setErr(null);
    try {
      const body: any = {
        slug: editing.slug,
        target: editing.target,
        oneTime: editing.oneTime,
        expireAt: editing.expireAt
      };
      const r = await fetch(`/api/links/${encodeURIComponent(editing.slug)}`, {
        method: "PATCH",
        headers: { "content-type": "application/json" },
        body: JSON.stringify(body)
      });
      const j = await r.json();
      if (r.ok) {
        setEditOpen(false);
        await fetchData();
      } else {
        setErr(j?.error || "Save failed");
      }
    } catch (e: any) {
      setErr(e?.message || "Save failed");
    } finally {
      setSaving(false);
    }
  }

  async function remove(slug: string) {
    if (!confirm(`Hapus ${slug}?`)) return;
    setDeleting(slug);
    setErr(null);
    try {
      const r = await fetch(`/api/links/${encodeURIComponent(slug)}`, { method: "DELETE" });
      const j = await r.json();
      if (r.ok) {
        await fetchData();
      } else {
        setErr(j?.error || "Delete failed");
      }
    } catch (e: any) {
      setErr(e?.message || "Delete failed");
    } finally {
      setDeleting(null);
    }
  }

  const totalPages = useMemo(() => data ? Math.max(1, Math.ceil(data.total / size)) : 1, [data, size]);

  return (
    <div className="space-y-3">
      <div className="toolbar">
        <input
          value={q}
          onChange={(e)=>{ setPage(1); setQ(e.target.value); }}
          placeholder="Search slug / target"
          className="input text-sm min-w-[240px] flex-1"
        />
        <select value={size} onChange={(e)=>{ setPage(1); setSize(Number(e.target.value)); }} className="input text-xs w-[80px]">
          {[10,20,50,100].map(n=> <option key={n} value={n}>{n}/page</option>)}
        </select>
        <button onClick={fetchData} className="btn-ghost flex items-center gap-2"><RefreshCw size={16}/> Refresh</button>
        {err && <span className="text-sm text-red-600">{err}</span>}
      </div>

      <div className="overflow-x-auto card p-6">
        <table className="table">
          <thead className="table-head">
            <tr>
              <th className="text-left px-3 py-2">Slug</th>
              <th className="text-left px-3 py-2">Target</th>
              <th className="text-left px-3 py-2">Clicks</th>
              <th className="text-left px-3 py-2">Expire</th>
              <th className="text-left px-3 py-2">OneTime</th>
              <th className="text-left px-3 py-2">Actions</th>
            </tr>
          </thead>
          <tbody>
            {loading ? (
              <tr><td className="px-3 py-3" colSpan={6}>Loading...</td></tr>
            ) : (data?.items.length ? data.items.map(row => (
              <tr key={row.id} className="border-t">
                <td className="px-3 py-2 font-mono">{row.slug}</td>
                <td className="px-3 py-2 max-w-[460px] truncate">{row.target}</td>
                <td className="px-3 py-2">{row.clicks}</td>
                <td className="px-3 py-2">{row.expireAt ? new Date(row.expireAt).toLocaleString() : "-"}</td>
                <td className="px-3 py-2">{row.oneTime ? "Yes" : "No"}</td>
                <td className="px-3 py-2 space-x-2">
                  <button onClick={()=>copyShort(row.slug)} className="btn-ghost" title={copying===row.slug?"Copied":"Copy"}><Copy size={16}/></button>
                  {role === 'ADMIN' && (<>
                    <button onClick={()=>openEdit(row)} className="btn-ghost" title="Edit"><Pencil size={16}/></button>
                    <button onClick={()=>remove(row.slug)} disabled={deleting===row.slug} className="btn-ghost" title="Delete"><Trash2 size={16}/></button>
                  </>)}
                </td>
              </tr>
            )) : (
              <tr><td className="px-3 py-3" colSpan={6}>No data</td></tr>
            ))}
          </tbody>
        </table>
      </div>

      {data && totalPages > 1 && (
        <div className="flex items-center gap-2">
          <button disabled={page<=1} onClick={()=>setPage(p=>p-1)} className="btn-ghost disabled:opacity-50">Prev</button>
          <div className="text-sm">Page {page} / {totalPages}</div>
          <button disabled={page>=totalPages} onClick={()=>setPage(p=>p+1)} className="btn-ghost disabled:opacity-50">Next</button>
        </div>
      )}

      {/* Edit modal */}
      <Modal
        open={editOpen}
        onClose={()=>setEditOpen(false)}
        title={editing ? `Edit: ${editing.slug}` : "Edit"}
        footer={<div className="flex items-center gap-2">
          <button onClick={()=>setEditOpen(false)} className="btn-ghost">Cancel</button>
          <button onClick={saveEdit} disabled={saving} className="btn-gradient">{saving?"Saving...":"Save"}</button>
        </div>}
      >
        {editing && (
          <div className="space-y-3">
            <div className="space-y-1">
              <label className="text-sm font-medium">Slug</label>
              <input
                value={editing.slug}
                onChange={(e)=>setEditing({...editing!, slug: e.target.value.toLowerCase()})}
                className="input"
              />
              <p className="text-xs text-neutral-500">Huruf kecil/angka/- (3–32). Mengubah slug akan update URL.</p>
            </div>
            <div className="space-y-1">
              <label className="text-sm font-medium">Target</label>
              <input
                value={editing.target}
                onChange={(e)=>setEditing({...editing!, target: e.target.value})}
                className="input"
              />
            </div>
            <div className="grid grid-cols-2 gap-3">
              <div className="space-y-1">
                <label className="text-sm font-medium">Expire At</label>
                <input
                  type="datetime-local"
                  value={editing.expireAt ? new Date(editing.expireAt).toISOString().slice(0,16) : ""}
                  onChange={(e)=>setEditing({...editing!, expireAt: e.target.value ? new Date(e.target.value).toISOString() : null})}
                  className="input"
                />
              </div>
              <div className="space-y-1">
                <label className="text-sm font-medium">One-time</label>
                <div className="flex items-center gap-2">
                  <input type="checkbox" checked={editing.oneTime} onChange={(e)=>setEditing({...editing!, oneTime: e.target.checked})} />
                  <span className="text-sm">Disable setelah 1 klik</span>
                </div>
              </div>
            </div>
          </div>
        )}
      </Modal>
    </div>
  );
}






