﻿"use client";

import { useEffect, useMemo, useState } from "react";
import { customAlphabet } from "nanoid";

const gen = customAlphabet("abcdefghijklmnopqrstuvwxyz0123456789", 6);

type Result = {
  slug: string;
  shortUrl: string;
  target: string;
  expireAt?: string;
  oneTime?: boolean;
} | { error: any };

function FieldLabel({ children }: { children: React.ReactNode }) {
  return <label className="text-sm font-medium">{children}</label>;
}

function Help({ children }: { children: React.ReactNode }) {
  return <p className="help">{children}</p>;
}

export default function ShortlinkForm() {
  const [domain, setDomain] = useState<string>("");
  const [target, setTarget] = useState<string>("");
  const [customSlug, setCustomSlug] = useState<string>("");
  const [suggested, setSuggested] = useState<string>("");
  const [password, setPassword] = useState<string>("");
  const [expireAt, setExpireAt] = useState<string>("");
  const [oneTime, setOneTime] = useState<boolean>(false);
  const [utmSource, setUtmSource] = useState<string>("");
  const [utmMedium, setUtmMedium] = useState<string>("");
  const [utmCampaign, setUtmCampaign] = useState<string>("");
  const [submitting, setSubmitting] = useState(false);
  const [res, setRes] = useState<Result | null>(null);
  const [copied, setCopied] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [showAdvanced, setShowAdvanced] = useState(false);

  useEffect(() => {
    if (typeof window !== "undefined") {
      const envBase = process?.env?.NEXT_PUBLIC_BASE_URL;
      setDomain(envBase || window.location.origin);
    }
    setSuggested(gen());
  }, []);

  const finalSlug = customSlug.trim() || suggested;
  const previewShort = useMemo(() => {
    if (!finalSlug || !domain) return "";
    return `${domain.replace(/\/$/, "")}/r/${finalSlug}`;
  }, [domain, finalSlug]);

  async function onSubmit(e: React.FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setSubmitting(true);
    setError(null);
    setRes(null);
    try {
      const payload: any = {
        target,
        customSlug: customSlug || undefined,
        password: password || undefined,
        oneTime,
        expireAt: expireAt ? new Date(expireAt).toISOString() : undefined,
        utm: {
          source: utmSource || undefined,
          medium: utmMedium || undefined,
          campaign: utmCampaign || undefined,
        }
      };
      const r = await fetch("/api/links", { method: "POST", headers: { "content-type": "application/json" }, body: JSON.stringify(payload) });
      const j = await r.json();
      setRes(j);
      if (!("error" in j)) {
        setSuggested(j.slug);
      } else {
        setError(typeof j.error === "string" ? j.error : "Gagal membuat shortlink");
      }
    } catch (err: any) {
      setError(err?.message || "Unexpected error");
    } finally {
      setSubmitting(false);
    }
  }

  async function copy(text: string) {
    try {
      await navigator.clipboard.writeText(text);
      setCopied(true);
      setTimeout(() => setCopied(false), 1200);
    } catch {}
  }

  function randomizeSlug() {
    const s = gen();
    setSuggested(s);
    if (!customSlug) setCustomSlug("");
  }

  const isValid = (() => {
    try {
      if (!target) return false;
      new URL(target);
      if (customSlug && !/^[a-z0-9-]{3,32}$/.test(customSlug)) return false;
      return true;
    } catch {
      return false;
    }
  })();

  return (
    <div className="space-y-6">
      <form onSubmit={onSubmit} className="card p-6 space-y-4">
        <div className="flex items-start justify-between gap-4">
          <div>
            <h3 className="text-lg font-semibold">Create Shortlink</h3>
            <p className="help">Isi tujuan, atur slug, lalu generate. Opsi lanjutan tersedia di bawah.</p>
          </div>
          <span className="text-xs rounded-full border px-2 py-1" style={{ borderColor: 'var(--line)', background: 'var(--panel)', color: '#EDEAFF' }}>Shortlinks</span>
        </div>

        <div className="space-y-1">
          <FieldLabel>Target URL</FieldLabel>
          <input name="target" value={target} onChange={(e) => setTarget(e.target.value)} required placeholder="https://example.com/landing" className="input" inputMode="url" />
          <Help>Tujuan asli yang akan dituju setelah redirect.</Help>
        </div>

        <div className="grid md:grid-cols-2 gap-3 items-end">
          <div className="space-y-1">
            <FieldLabel>Custom Slug (opsional)</FieldLabel>
            <input name="customSlug" value={customSlug} onChange={(e) => setCustomSlug(e.target.value.toLowerCase())} placeholder="promo-oktober" className="input" />
            <Help>Huruf kecil, angka, strip. 3–32 karakter.</Help>
          </div>
          <div className="space-y-1">
            <label className="label">Preview</label>
            <div className="flex items-center gap-2">
              <input readOnly value={previewShort || ""} className="input text-sm opacity-100" placeholder="http://localhost:3000/r/slug" />
              <button type="button" onClick={randomizeSlug} className="btn-ghost" title="Random slug" aria-label="Random slug">↻</button>
            </div>
            <p className="help">Slug dipakai: <code className="bg-[#1c1536] px-1 rounded">{finalSlug || "-"}</code></p>
          </div>
        </div>

        <div>
          <button type="button" onClick={() => setShowAdvanced(s => !s)} className="text-sm underline underline-offset-2">
            {showAdvanced ? "Sembunyikan opsi lanjutan" : "Tampilkan opsi lanjutan"}
          </button>
        </div>

        {showAdvanced && (
          <div className="grid gap-5 md:grid-cols-2">
            <div className="space-y-3">
              <div className="space-y-1">
                <FieldLabel>Password (opsional)</FieldLabel>
                <input name="password" value={password} onChange={(e) => setPassword(e.target.value)} type="password" placeholder="Password" className="input" />
                <Help>Pengunjung harus memasukkan password untuk membuka link.</Help>
              </div>
              <div className="space-y-1">
                <FieldLabel>Expire At (opsional)</FieldLabel>
                <input name="expireAt" value={expireAt} onChange={(e) => setExpireAt(e.target.value)} type="datetime-local" className="input" />
                <Help>Set waktu kadaluarsa. Setelah lewat akan tidak aktif.</Help>
              </div>
              <div className="flex items-center gap-2">
                <input id="oneTime" name="oneTime" type="checkbox" checked={oneTime} onChange={(e) => setOneTime(e.target.checked)} />
                <label htmlFor="oneTime" className="text-sm">One-time link (disable setelah 1 klik)</label>
              </div>
            </div>

            <div className="space-y-3">
              <div className="space-y-1">
                <FieldLabel>UTM Parameters (opsional)</FieldLabel>
                <div className="grid grid-cols-3 gap-2">
                  <input value={utmSource} onChange={(e)=>setUtmSource(e.target.value)} placeholder="utm_source" className="input" />
                  <input value={utmMedium} onChange={(e)=>setUtmMedium(e.target.value)} placeholder="utm_medium" className="input" />
                  <input value={utmCampaign} onChange={(e)=>setUtmCampaign(e.target.value)} placeholder="utm_campaign" className="input" />
                </div>
                <Help>UTM akan otomatis ditempel ke target URL.</Help>
              </div>
              <div className="space-y-1">
                <FieldLabel>Base URL</FieldLabel>
                <input value={domain} onChange={(e)=>setDomain(e.target.value)} className="input" />
                <Help>Ambil dari <code>NEXT_PUBLIC_BASE_URL</code> kalau ada. Bisa diubah untuk multi-domain.</Help>
              </div>
            </div>
          </div>
        )}

        <div className="flex items-center gap-3 pt-2">
          <button disabled={!isValid || submitting} className={`btn-gradient ${(!isValid || submitting) ? "opacity-50" : ""}`}>
            {submitting ? "Generating..." : "Generate"}
          </button>
          {error && <span className="text-sm text-red-500">{error}</span>}
        </div>
      </form>

      {res && !("error" in res) && (
        <div className="card p-6 space-y-3">
          <div className="flex items-center justify-between">
            <h4 className="font-semibold">Shortlink Generated</h4>
            <span className="chip chip-purple">Success</span>
          </div>
          <div className="flex flex-wrap items-center gap-2">
            <a className="underline break-all" style={{ color: '#EDEAFF' }} href={(res as any).shortUrl} target="_blank" rel="noreferrer">{(res as any).shortUrl}</a>
            <button onClick={()=>copy((res as any).shortUrl)} className="btn-ghost">Copy</button>
            {copied && <span className="help">Copied!</span>}
          </div>
          <div className="help break-all">Target: {(res as any).target}</div>
          <div className="help">{(res as any).oneTime ? "One-time • " : ""}{(res as any).expireAt ? `Expires: ${new Date((res as any).expireAt).toLocaleString()}` : "No expiry"}</div>
        </div>
      )}
    </div>
  );
}
