"use client";

import { useState } from "react";

export default function QuickCreate() {
  const [target, setTarget] = useState("");
  const [customSlug, setCustomSlug] = useState("");
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [result, setResult] = useState<{ shortUrl: string; slug: string } | null>(null);
  const [copied, setCopied] = useState(false);

  async function onSubmit(e: React.FormEvent) {
    e.preventDefault();
    setLoading(true);
    setError(null);
    setResult(null);
    try {
      const body: any = { target };
      if (customSlug) body.customSlug = customSlug;
      const r = await fetch("/api/links", {
        method: "POST",
        headers: { "content-type": "application/json" },
        body: JSON.stringify(body),
      });
      const j = await r.json();
      if (!r.ok) throw new Error(j?.error || "Gagal membuat shortlink");
      setResult({ shortUrl: j.shortUrl, slug: j.slug });
      setTarget("");
      setCustomSlug("");
    } catch (e: any) {
      setError(e?.message || "Gagal membuat shortlink");
    } finally {
      setLoading(false);
    }
  }

  async function copy(text: string) {
    try {
      await navigator.clipboard.writeText(text);
      setCopied(true);
      setTimeout(() => setCopied(false), 1200);
    } catch {}
  }

  return (
    <div className="card p-6 space-y-4">
      <div className="text-lg font-semibold">Quick Create</div>
      <form onSubmit={onSubmit} className="space-y-3">
        <div className="space-y-1">
          <label className="label">Target URL</label>
          <input className="input" value={target} onChange={(e)=>setTarget(e.target.value)} placeholder="https://example.com" />
        </div>
        <div className="space-y-1">
          <label className="label">Custom Slug (opsional)</label>
          <input className="input" value={customSlug} onChange={(e)=>setCustomSlug(e.target.value.toLowerCase())} placeholder="promo-oktober" />
        </div>
        {error && <div className="text-sm text-rose-400">{error}</div>}
        <button className={`btn-gradient w-full ${loading ? 'opacity-50' : ''}`} disabled={loading}>
          {loading ? 'Generating...' : 'Generate'}
        </button>
      </form>
      {result && (
        <div className="space-y-2">
          <div className="help">Shortlink:</div>
          <div className="flex items-center gap-2">
            <a className="underline break-all" style={{ color: '#EDEAFF' }} href={result.shortUrl} target="_blank" rel="noreferrer">{result.shortUrl}</a>
            <button className="btn-ghost" onClick={()=>copy(result.shortUrl)}>{copied ? 'Copied' : 'Copy'}</button>
          </div>
        </div>
      )}
    </div>
  );
}
