﻿import prisma from "@/lib/prisma";
import Link from "next/link";
import QuickCreate from "@/components/QuickCreate";

export default async function Dashboard() {
  const now = new Date();
  const [agg, activeCount, expiredCount, oneTimeCount, recent] = await Promise.all([
    prisma.link.aggregate({ _count: true, _sum: { clicks: true } }),
    prisma.link.count({ where: { OR: [{ expireAt: null }, { expireAt: { gt: now } }] } }),
    prisma.link.count({ where: { expireAt: { lt: now } } }),
    prisma.link.count({ where: { oneTime: true } }),
    prisma.link.findMany({
      orderBy: { createdAt: "desc" },
      take: 5,
      select: { id: true, slug: true, target: true, clicks: true, expireAt: true, createdAt: true, oneTime: true },
    }),
  ]);

  const totalLinks = agg._count || 0;
  const totalClicks = agg._sum.clicks || 0;

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-semibold">Dashboard</h2>
          <p className="help">Ringkasan activity dan aksi cepat.</p>
        </div>
      </div>

      <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
        <div className="card p-5">
          <div className="text-xs" style={{ color: "var(--muted)" }}>
            Total Links
          </div>
          <div className="mt-1 text-2xl font-semibold">{totalLinks}</div>
        </div>
        <div className="card p-5">
          <div className="text-xs" style={{ color: "var(--muted)" }}>
            Total Clicks
          </div>
          <div className="mt-1 text-2xl font-semibold">{totalClicks}</div>
        </div>
        <div className="card p-5">
          <div className="text-xs" style={{ color: "var(--muted)" }}>
            Active
          </div>
          <div className="mt-1 text-2xl font-semibold">{activeCount}</div>
        </div>
        <div className="card p-5">
          <div className="text-xs" style={{ color: "var(--muted)" }}>
            Expired
          </div>
          <div className="mt-1 text-2xl font-semibold">{expiredCount}</div>
        </div>
      </div>

      <div className="grid gap-6 lg:grid-cols-3">
        <div className="lg:col-span-2 card p-6">
          <div className="flex items-center justify-between mb-3">
            <div className="font-semibold">Recent Shortlinks</div>
            <Link href="/shortlinks" className="btn-ghost">
              Kelola Shortlink
            </Link>
          </div>
          <div className="overflow-x-auto">
            <table className="table">
              <thead className="table-head">
                <tr>
                  <th>Slug</th>
                  <th>Target</th>
                  <th>Clicks</th>
                  <th>Expire</th>
                  <th>Created</th>
                </tr>
              </thead>
              <tbody>
                {recent.map((row) => (
                  <tr key={row.id}>
                    <td className="px-3 py-2 font-mono">{row.slug}</td>
                    <td className="px-3 py-2 max-w-[460px] truncate">{row.target}</td>
                    <td className="px-3 py-2">{row.clicks}</td>
                    <td className="px-3 py-2">{row.expireAt ? new Date(row.expireAt).toLocaleString() : "-"}</td>
                    <td className="px-3 py-2">{new Date(row.createdAt).toLocaleDateString()}</td>
                  </tr>
                ))}
                {recent.length === 0 && (
                  <tr>
                    <td className="px-3 py-3" colSpan={5}>
                      No data
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        </div>
        <div>
          <QuickCreate />
        </div>
      </div>
    </div>
  );
}
