import { NextRequest, NextResponse } from "next/server";
import { getServerSession } from "next-auth";
import { authOptions } from "@/lib/auth";
import path from "path";
import { promises as fs } from "fs";

type ToolItem = {
  name: string;
  slug: string;
  path: string;
  icon?: string;
  cate?: string;
};

const toolsDir = path.join(process.cwd(), "public", "tools");
const indexPath = path.join(toolsDir, "index.json");

async function readIndex(): Promise<ToolItem[]> {
  try {
    const buf = await fs.readFile(indexPath, "utf8");
    const data = JSON.parse(buf);
    return Array.isArray(data) ? data : [];
  } catch {
    return [];
  }
}

async function writeIndex(items: ToolItem[]) {
  await fs.mkdir(toolsDir, { recursive: true });
  await fs.writeFile(indexPath, JSON.stringify(items, null, 2), "utf8");
}

function slugify(s: string) {
  return s
    .toLowerCase()
    .trim()
    .replace(/[^a-z0-9\-\s]/g, "")
    .replace(/\s+/g, "-")
    .replace(/-+/g, "-")
    .slice(0, 64);
}

export async function GET() {
  const items = await readIndex();
  return NextResponse.json(items);
}

export async function POST(req: NextRequest) {
  const session = await getServerSession(authOptions);
  if (session?.user?.role !== "ADMIN") {
    return NextResponse.json({ error: "Unauthorized" }, { status: 403 });
  }

  const form = await req.formData();
  const name = String(form.get("name") || "").trim();
  const category = String(form.get("cate") || "").trim() || undefined;
  const customSlug = String(form.get("slug") || "").trim();
  const file = form.get("file") as File | null;

  if (!name || !file) {
    return NextResponse.json({ error: "Missing name or file" }, { status: 400 });
  }

  let slug = customSlug ? slugify(customSlug) : slugify(name);
  if (!slug) slug = `tool-${Date.now()}`;

  const items = await readIndex();
  // ensure unique slug
  let base = slug;
  let i = 1;
  while (items.some((t) => t.slug === slug)) {
    slug = `${base}-${i++}`;
  }

  // save file
  await fs.mkdir(toolsDir, { recursive: true });
  const filePath = path.join(toolsDir, `${slug}.html`);
  const arrayBuffer = await file.arrayBuffer();
  await fs.writeFile(filePath, Buffer.from(arrayBuffer));

  const item: ToolItem = {
    name,
    slug,
    path: `/tools/${slug}.html`,
    cate: category,
  };
  items.push(item);
  await writeIndex(items);

  return NextResponse.json(item, { status: 201 });
}

