import { NextRequest, NextResponse } from "next/server";
import { authSession } from "@/lib/auth";
import { z } from "zod";
import { PrismaClient } from "@prisma/client";
import { genSlug, withUTM, hashPassword } from "@/lib/shorty";

const prisma = new PrismaClient();

// naive in-memory rate-limit per process
const buckets = new Map<string, {count: number; ts: number}>();
function rl(ip: string, limit = 60, windowMs = 60_000) {
  const now = Date.now();
  const b = buckets.get(ip) ?? { count: 0, ts: now };
  if (now - b.ts > windowMs) { b.count = 0; b.ts = now; }
  b.count++;
  buckets.set(ip, b);
  return b.count <= limit;
}

// ---------- GET /api/links  (list with optional q, page, size) ----------
export async function GET(req: NextRequest) {
  const { searchParams } = new URL(req.url);
  const q = (searchParams.get("q") || "").toLowerCase().trim();
  const page = Number(searchParams.get("page") || "1");
  const size = Math.min(100, Math.max(1, Number(searchParams.get("size") || "20")));
  const skip = (page - 1) * size;

  const where = q ? {
    OR: [
      { slug: { contains: q } },
      { target: { contains: q } },
    ]
  } : {};

  const [total, items] = await Promise.all([
    prisma.link.count({ where }),
    prisma.link.findMany({
      where,
      orderBy: { createdAt: "desc" },
      skip,
      take: size,
      select: { id: true, slug: true, target: true, clicks: true, oneTime: true, expireAt: true, createdAt: true, updatedAt: true }
    })
  ]);

  return NextResponse.json({ page, size, total, items });
}

// ---------- POST /api/links  (create) ----------
const Body = z.object({
  target: z.string().url(),
  customSlug: z.string().min(3).max(32).regex(/^[a-z0-9-]+$/).optional(),
  password: z.string().min(4).max(64).optional(),
  expireAt: z.coerce.date().optional(),
  oneTime: z.boolean().optional(),
  utm: z.object({
    source: z.string().optional(),
    medium: z.string().optional(),
    campaign: z.string().optional(),
  }).optional()
});

export async function POST(req: NextRequest) {
  const ip = req.headers.get("x-forwarded-for") ?? "local";
  if (!rl(ip)) return NextResponse.json({ error: "Rate limited" }, { status: 429 });

  const json = await req.json().catch(() => null);
  const parsed = Body.safeParse(json);
  if (!parsed.success) {
    return NextResponse.json({ error: parsed.error.flatten() }, { status: 400 });
  }

  const { target, customSlug, password, expireAt, oneTime, utm } = parsed.data;

  const session = await authSession();
  const ownerId = session?.user?.id as string | undefined;

  const finalTarget = utm ? withUTM(target, utm) : target;

  let slug = customSlug ?? genSlug();

  if (!customSlug) {
    for (let i = 0; i < 5; i++) {
      const exists = await prisma.link.findUnique({ where: { slug } });
      if (!exists) break;
      slug = genSlug();
    }
  } else {
    const exists = await prisma.link.findUnique({ where: { slug } });
    if (exists) return NextResponse.json({ error: "Slug already taken." }, { status: 409 });
  }

  const passwordHash = await hashPassword(password);

  const link = await prisma.link.create({
    data: { slug, target: finalTarget, passwordHash, expireAt: expireAt ?? null, oneTime: !!oneTime, ownerId: ownerId ?? null }
  });

  return NextResponse.json({
    slug: link.slug,
    shortUrl: `${process.env.NEXT_PUBLIC_BASE_URL ?? ""}/r/${link.slug}`,
    target: link.target,
    expireAt: link.expireAt,
    oneTime: link.oneTime
  });
}
