"use client";

import { useEffect, useMemo, useState } from "react";
import Modal from "@/components/Modal";

type ToolItem = { name: string; slug: string; path: string; icon?: string; cate?: string };

export default function AdminToolsPage() {
  const [name, setName] = useState("");
  const [slug, setSlug] = useState("");
  const [cate, setCate] = useState("");
  const [file, setFile] = useState<File | null>(null);
  const [submitting, setSubmitting] = useState(false);
  const [result, setResult] = useState<any>(null);
  const [error, setError] = useState<string | null>(null);
  const [items, setItems] = useState<ToolItem[]>([]);
  const [q, setQ] = useState("");
  const [edit, setEdit] = useState<ToolItem | null>(null);
  const [editOpen, setEditOpen] = useState(false);
  const [editFile, setEditFile] = useState<File | null>(null);
  const [delOpen, setDelOpen] = useState(false);
  const [delTarget, setDelTarget] = useState<ToolItem | null>(null);

  async function onSubmit(e: React.FormEvent) {
    e.preventDefault();
    setSubmitting(true);
    setError(null);
    setResult(null);
    try {
      if (!name || !file) { setError("Nama dan file wajib diisi"); return; }
      const fd = new FormData();
      fd.set("name", name);
      if (slug) fd.set("slug", slug);
      if (cate) fd.set("cate", cate);
      fd.set("file", file);
      const r = await fetch("/api/tools", { method: "POST", body: fd });
      const j = await r.json();
      if (!r.ok) throw new Error(j?.error || "Gagal menambah tool");
      setResult(j);
      setName(""); setSlug(""); setCate(""); setFile(null);
      await loadItems();
    } catch (err: any) {
      setError(err?.message || "Gagal menambah tool");
    } finally {
      setSubmitting(false);
    }
  }

  async function loadItems() {
    try {
      const r = await fetch("/api/tools");
      const j = await r.json();
      if (Array.isArray(j)) setItems(j);
    } catch {}
  }

  useEffect(() => { loadItems(); }, []);

  const filtered = useMemo(() => {
    const s = q.trim().toLowerCase();
    if (!s) return items;
    return items.filter(i => i.name.toLowerCase().includes(s) || i.slug.includes(s) || i.cate?.toLowerCase().includes(s));
  }, [items, q]);

  function askDelete(it: ToolItem) {
    setDelTarget(it);
    setDelOpen(true);
  }

  async function confirmDelete() {
    if (!delTarget) return;
    const r = await fetch(`/api/tools/${encodeURIComponent(delTarget.slug)}`, { method: "DELETE" });
    if (r.ok) {
      setDelOpen(false);
      setDelTarget(null);
      await loadItems();
    }
  }

  function openEdit(it: ToolItem) {
    setEdit({ ...it });
    setEditFile(null);
    setEditOpen(true);
  }

  async function saveEdit() {
    if (!edit) return;
    const fd = new FormData();
    fd.set("name", edit.name);
    if (edit.cate) fd.set("cate", edit.cate);
    if (edit.slug) fd.set("slug", edit.slug);
    if (editFile) fd.set("file", editFile);
    const r = await fetch(`/api/tools/${encodeURIComponent(edit.slug)}`, { method: "PATCH", body: fd });
    if (r.ok) {
      setEditOpen(false);
      await loadItems();
    }
  }

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-semibold">Manage Tools</h2>
          <p className="help">Upload file HTML dan beri nama, tool otomatis muncul di sidebar.</p>
        </div>
      </div>

      <form onSubmit={onSubmit} className="card p-6 space-y-4">
        <div className="grid md:grid-cols-2 gap-4">
          <div className="space-y-1">
            <label className="label">Nama Tool</label>
            <input className="input" value={name} onChange={(e)=>setName(e.target.value)} placeholder="Contoh: SERP Checker" />
          </div>
          <div className="space-y-1">
            <label className="label">Slug (opsional)</label>
            <input className="input" value={slug} onChange={(e)=>setSlug(e.target.value.toLowerCase())} placeholder="serp-checker" />
            <p className="help">Kosongkan untuk otomatis dari nama.</p>
          </div>
          <div className="space-y-1">
            <label className="label">Kategori (opsional)</label>
            <input className="input" value={cate} onChange={(e)=>setCate(e.target.value)} placeholder="SEO / Network" />
          </div>
          <div className="space-y-1">
            <label className="label">File HTML</label>
            <input className="input" type="file" accept=".html,text/html" onChange={(e)=>setFile(e.target.files?.[0] || null)} />
            <p className="help">File akan disimpan ke <code>/public/tools/&lt;slug&gt;.html</code></p>
          </div>
        </div>

        <div className="flex items-center gap-3">
          <button disabled={submitting || !name || !file} className={`btn-gradient ${submitting ? 'opacity-50' : ''}`}>{submitting ? 'Uploading...' : 'Upload Tool'}</button>
          {error && <span className="text-sm text-red-500">{error}</span>}
        </div>

        {result && (
          <div className="help">
            Berhasil: <a className="underline" href={`/tools/${result.slug}`} target="_blank">/tools/{result.slug}</a>
          </div>
        )}
      </form>

      <div className="card p-6 space-y-3">
        <div className="flex items-center justify-between">
          <div className="font-semibold">Daftar Tools</div>
          <input value={q} onChange={(e)=>setQ(e.target.value)} placeholder="Cari tools" className="input w-64" />
        </div>
        <div className="overflow-x-auto">
          <table className="table">
            <thead className="table-head">
              <tr>
                <th>Nama</th>
                <th>Slug</th>
                <th>Kategori</th>
                <th>Path</th>
                <th>Actions</th>
              </tr>
            </thead>
            <tbody>
              {filtered.map(it => (
                <tr key={it.slug}>
                  <td className="px-3 py-2">{it.name}</td>
                  <td className="px-3 py-2 font-mono">{it.slug}</td>
                  <td className="px-3 py-2">{it.cate ?? '-'}</td>
                  <td className="px-3 py-2"><a className="underline" href={it.path} target="_blank" rel="noreferrer">{it.path}</a></td>
                  <td className="px-3 py-2 space-x-2">
                    <button className="btn-ghost" onClick={()=>openEdit(it)}>Edit</button>
                    <button className="btn-ghost" onClick={()=>askDelete(it)}>Delete</button>
                  </td>
                </tr>
              ))}
              {filtered.length === 0 && (
                <tr><td className="px-3 py-3" colSpan={5}>Tidak ada data</td></tr>
              )}
            </tbody>
          </table>
        </div>
      </div>

      <Modal open={editOpen} onClose={()=>setEditOpen(false)} title="Edit Tool">
        {edit && (
          <div className="space-y-3">
            <div className="grid md:grid-cols-2 gap-3">
              <div className="space-y-1">
                <label className="label">Nama</label>
                <input className="input" value={edit.name} onChange={(e)=>setEdit({...edit, name: e.target.value})} />
              </div>
              <div className="space-y-1">
                <label className="label">Slug</label>
                <input className="input" value={edit.slug} onChange={(e)=>setEdit({...edit, slug: e.target.value.toLowerCase()})} />
                {(!/^[a-z0-9-]{3,64}$/.test(edit.slug)) && (
                  <p className="help">Slug harus 3–64 karakter, huruf kecil/angka/-</p>
                )}
              </div>
              <div className="space-y-1">
                <label className="label">Kategori</label>
                <input className="input" value={edit.cate ?? ''} onChange={(e)=>setEdit({...edit, cate: e.target.value})} />
              </div>
              <div className="space-y-1">
                <label className="label">Ganti File HTML (opsional)</label>
                <input className="input" type="file" accept=".html,text/html" onChange={(e)=>setEditFile(e.target.files?.[0] || null)} />
              </div>
            </div>
            <div className="flex items-center justify-end gap-2">
              <button className="btn-ghost" onClick={()=>setEditOpen(false)}>Batal</button>
              <button className="btn-gradient disabled:opacity-50" onClick={saveEdit} disabled={!/^[a-z0-9-]{3,64}$/.test(edit.slug)}>Simpan</button>
            </div>
          </div>
        )}
      </Modal>

      <Modal open={delOpen} onClose={()=>setDelOpen(false)} title="Hapus Tool">
        <div className="space-y-3">
          <p>Yakin ingin menghapus tool <b>{delTarget?.name}</b> ({delTarget?.slug})? Tindakan ini akan menghapus file HTML terkait.</p>
          <div className="flex items-center justify-end gap-2">
            <button className="btn-ghost" onClick={()=>setDelOpen(false)}>Batal</button>
            <button className="btn-gradient" onClick={confirmDelete}>Hapus</button>
          </div>
        </div>
      </Modal>
    </div>
  );
}
