﻿# Admin Tools Panel (Private)

Darkâ€‘purple admin panel to manage your inâ€‘house HTML tools + shortlink service (CRUD), built with **Next.js (App Router) + Tailwind + Prisma + NextAuth v4 (JWT)**.  
Optimized for **personal/private** use: login first, then sidebar + tools.

---

## Features
- **Private auth** (Credentials) with **roles**: `ADMIN` (full CRUD) & `OPERATOR` (read/create).
- **Shortlinks**: create, list, search, edit (modal), delete, copy.
- **Advanced options**: one-time link, expire time, optional password, UTM auto-append.
- **Owner tracking**: new links record `ownerId` from session.
- **UI/UX**: darkâ€‘purple theme, gradient buttons, iconâ€‘only actions, clean table.
- **Global protection** via middleware: `/signin` only public; everything else requires login.
- **Dashboard + sidebar** with sample tool entries (edit `lib/tools.ts`).

---

## ðŸ§± Tech Stack
- **Next.js** (App Router)
- **Tailwind CSS**
- **NextAuth v4** (Credentials, JWT sessions)
- **Prisma** ORM
- **SQLite (dev)** â†’ **PostgreSQL (recommended for prod)**

---

## ðŸ“ Project Structure (key)
```
app/
  api/
    auth/[...nextauth]/route.ts     # NextAuth handler (v4)
    links/route.ts                   # GET list, POST create
    links/[slug]/route.ts            # GET, PATCH, DELETE
    whoami/route.ts                  # session role (client use)
  dashboard/page.tsx                 # landing after login
  shortlinks/page.tsx                # shortlink form + table
  signin/page.tsx                    # server page â†’ redirects if authed
  signin/signin-client.tsx           # client login UI
  layout.tsx, page.tsx               # app shell, root redirect
components/
  Sidebar.tsx, Modal.tsx, ShortlinkForm.tsx, ShortlinkTable.tsx
lib/
  auth.ts                            # NextAuth config + getServerSession()
  tools.ts                           # Register your HTML tools here
prisma/
  schema.prisma                      # DB schema
  seed.ts                            # seed admin/operator
public/tools/                        # put your static HTML tools here (optional)
app/globals.css                      # theme + UI utilities
```

---

## ðŸš€ Getting Started (Local Dev)

> Requires Node.js **18+**

1) Install deps
```bash
npm i
```

2) Environment
```bash
# copy defaults
cp .env.local.example .env.local
# open and set:
# - NEXTAUTH_URL=http://localhost:3000
# - NEXTAUTH_SECRET=some-long-random-string
# - NEXT_PUBLIC_BASE_URL=http://localhost:3000
```

3) DB migrate & generate client
```bash
npx prisma migrate dev -n init
```

4) Seed users
```bash
npm run seed
# default:
#  ADMIN    admin@example.com / admin123
#  OPERATOR op@example.com    / operator123
```

5) Run
```bash
npm run dev
# http://localhost:3000/signin
```

---

## ðŸ” Roles
- **ADMIN**: create, edit, delete shortlinks.
- **OPERATOR**: create + view + copy (no edit/delete).
- Middleware blocks `PATCH/DELETE /api/links*` for nonâ€‘ADMIN.

---

## Environment Variables

`.env.local`
```
# NextAuth (v4)
NEXTAUTH_URL=http://localhost:3000
NEXTAUTH_SECRET=changeme-this-should-be-random

# Base URL used to render short URLs in UI
NEXT_PUBLIC_BASE_URL=http://localhost:3000

# (Prod) switch to Postgres:
# DATABASE_URL=postgresql://user:password@host:5432/dbname?schema=public
```

`.env.local` also supports seeding overrides:
```
INIT_ADMIN_EMAIL=admin@example.com
INIT_ADMIN_PASSWORD=admin123
INIT_OPERATOR_EMAIL=op@example.com
INIT_OPERATOR_PASSWORD=operator123
```

---

## ðŸ› ï¸ Common Commands
```bash
# Prisma
npx prisma generate
npx prisma studio          # DB browser
npx prisma migrate dev -n change

# Seeds
npm run seed
```

---

## ðŸ§ª Shortlink API (quick reference)

- `GET  /api/links?q=&page=&size=` â†’ list + search
- `POST /api/links` body:
```json
{
  "target": "https://example.com",
  "customSlug": "promo-oktober",
  "password": "optional",
  "expireAt": "2025-12-31T23:59:59.000Z",
  "oneTime": false,
  "utm": { "source": "ads", "medium": "cpc", "campaign": "oct" }
}
```
- `GET    /api/links/[slug]`
- `PATCH  /api/links/[slug]` â†’ `{ slug?, target?, oneTime?, expireAt?: string|null }`
- `DELETE /api/links/[slug]`

---

## ðŸŽ¨ Theming

Edit **`app/globals.css`** â†’ change CSS variables inside `:root`:
```
--bg, --panel, --panel-2, --text-muted, --hairline, --grad-from, --grad-mid, --grad-to
```
Utility classes used across the app: `.card`, `.input`, `.btn-gradient`, `.btn-ghost`, `.btn-icon`, `.table`, `.label`, `.help`, `.chip`.

---

## ðŸ“¦ Deployment

### Option A â€” Vercel (recommended, with Postgres)
SQLite is not writable on serverless environments. Use a managed Postgres (e.g. **Neon**, **Supabase**).

1) Create a Postgres DB and get `DATABASE_URL`.
2) In Vercel project â†’ **Settings â†’ Environment Variables**:
```
DATABASE_URL=postgresql://user:pass@host:5432/db?schema=public
NEXTAUTH_URL=https://your-domain.tld
NEXTAUTH_SECRET=your-random-string
NEXT_PUBLIC_BASE_URL=https://your-domain.tld
INIT_ADMIN_EMAIL=admin@your.tld         # optional
INIT_ADMIN_PASSWORD=StrongPass!         # optional
INIT_OPERATOR_EMAIL=ops@your.tld        # optional
INIT_OPERATOR_PASSWORD=StrongPass!      # optional
```
3) Push code to Git repo and **Import** the repo to Vercel.
4) After first deploy, run migrations (Vercel CLI or GitHub Action):
```bash
# locally, pointing to the prod DB:
DATABASE_URL="postgresql://..." npx prisma migrate deploy
# and (optional) seed:
DATABASE_URL="postgresql://..." npm run seed
```
5) Open your app URL â†’ `/signin`

> If you switch from SQLite to Postgres, update `prisma/schema.prisma` datasource to `provider = "postgresql"` and run a new migration before deploy.

---

### Option B â€” Docker (VPS / selfâ€‘hosted)
Use Postgres for persistence.

**Dockerfile**
```dockerfile
# 1) Build
FROM node:20-alpine AS builder
WORKDIR /app
COPY package*.json ./
RUN npm ci
COPY . .
RUN npm run build

# 2) Run
FROM node:20-alpine
WORKDIR /app
ENV NODE_ENV=production
COPY --from=builder /app .
RUN npm ci --omit=dev
EXPOSE 3000
CMD ["npm", "start"]
```

**docker-compose.yml**
```yaml
version: "3.9"
services:
  db:
    image: postgres:16
    environment:
      POSTGRES_USER: app
      POSTGRES_PASSWORD: app
      POSTGRES_DB: admin_tools
    volumes:
      - dbdata:/var/lib/postgresql/data
    ports: ["5432:5432"]

  web:
    build: .
    environment:
      DATABASE_URL: postgres://app:app@db:5432/admin_tools
      NEXTAUTH_URL: http://localhost:3000
      NEXTAUTH_SECRET: changeme
      NEXT_PUBLIC_BASE_URL: http://localhost:3000
    ports: ["3000:3000"]
    depends_on: [db]
volumes:
  dbdata:
```

**Init DB in container**
```bash
docker compose run --rm web npx prisma migrate deploy
docker compose run --rm web npm run seed
docker compose up -d
```

---

### Option C â€” Bare metal / Node on VPS
1) Install Node 20.
2) Install Postgres and create DB; set `DATABASE_URL` in `.env`.
3) `npm ci && npm run build`
4) `npx prisma migrate deploy && npm run seed`
5) Run with PM2:
```bash
npm i -g pm2
pm2 start "npm start" --name admin-tools
pm2 save && pm2 startup
```

---

## ðŸ§¯ Troubleshooting

**Credentials error: â€œSignin with credentials only supported if JWT strategy is enabledâ€**  
â†’ In `lib/auth.ts`, set `session: { strategy: "jwt" }` and restart dev server.

**Prisma SQLite errors (@db.Text / enums)**  
â†’ Remove `@db.Text` annotations; use `String` only. Avoid enum on SQLite; use `String` with default.

**Module not found for signin-client**  
â†’ Ensure `app/signin/signin-client.tsx` exists or import from `@/components/SignInClient`.

**White inputs on dark theme**  
â†’ Ensure `.input` uses dark background in `globals.css` and avoid `bg-white` in components.

---

## ðŸ”’ Security Notes
- Credentials are simpleâ€”use strong passwords and rotate `NEXTAUTH_SECRET` in prod.
- Consider enabling HTTPS everywhere (Vercel handles this).
- For multiâ€‘user setups, filter shortlinks by `ownerId` for operators (currently easy to add).

---

## License
Private/internal use.

---

## Updates (UI + Tools Manager)

- Tools Manager (Admin): upload HTML tool (auto saved to `public/tools/<slug>.html`), index managed in `public/tools/index.json`. Edit (name/slug/cate/file) and delete supported.
- Sidebar: responsive, collapsible, icon-only on small screens. Desktop toggle (close/menu_open). Floating tooltips when collapsed.
- Dashboard: summary cards + recent shortlinks + quick create form.
- Sign-in page: branded header with glow logo, clean form, dark purple theme.

API endpoints:
- `POST /api/tools` (multipart: name, slug?, cate?, file)
- `GET /api/tools`
- `PATCH /api/tools/[slug]` and `DELETE /api/tools/[slug]`

---

## Deploy to cPanel (Node.js App)

cPanel (Passenger) can run Next.js server. Recommended use Postgres in production.

1) Environment variables (`.env.production`)
```
DATABASE_URL=postgresql://user:pass@host:5432/db?schema=public
NEXTAUTH_URL=https://your-domain.tld
NEXTAUTH_SECRET=your-random-string
NEXT_PUBLIC_BASE_URL=https://your-domain.tld
```

2) Upload project and build
```
cd ~/apps/admin-tools
npm ci
npx prisma migrate deploy
npm run build
```

3) Setup Node.js App in cPanel
- Create Application → App root: `~/apps/admin-tools`, Node 18/20
- Ensure `package.json` has: `"start": "next start -p $PORT"`
- Start/restart the app (Passenger will bind to the assigned port)

4) Point domain/subdomain to the app per cPanel docs and open `/signin`.

Notes
- SQLite may work on shared hosting but Postgres is strongly recommended.
- If the app fails to boot, check Node version, memory limits, and logs.

---

## Deploy to aaPanel (PM2 + Nginx)

1) Install Node and upload project
```
cd /www/wwwroot/admin-tools
npm ci
npx prisma migrate deploy
npm run build
```

2) PM2 Manager
- Add app: Name `admin-tools`, Script `npm`, Args `start`, Working dir set to project
- Add env vars: `DATABASE_URL`, `NEXTAUTH_URL`, `NEXTAUTH_SECRET`, `NEXT_PUBLIC_BASE_URL`

3) Nginx reverse proxy
```
location / {
  proxy_pass http://127.0.0.1:3000;
  proxy_set_header Host $host;
  proxy_set_header X-Real-IP $remote_addr;
  proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
  proxy_set_header X-Forwarded-Proto $scheme;
}
```

4) SSL via Let’s Encrypt, then visit `/signin`.

Tips
- Change port by setting `PORT` in PM2 env and updating Nginx proxy.
- Use `pm2 logs` to debug startup/runtime issues.

