# 🧠 ai.md – Developer AI Brief

## 🎯 Project Purpose
Private **Admin Tools Panel** for managing internal HTML tools and shortlink generator.  
Built with **Next.js (App Router)**, **Tailwind CSS**, **NextAuth (JWT)**, and **Prisma**.  
Target use: personal or small-team dashboard with role-based access.

---

## ⚙️ Core Architecture
- **Frontend:** Next.js App Router + React Server Components.
- **Styling:** TailwindCSS + custom dark-purple theme.
- **Database:** Prisma ORM (SQLite dev / Postgres prod).
- **Auth:** NextAuth v4 (credentials only, JWT session).
- **Session roles:** `ADMIN`, `OPERATOR` (string-based for easy extension).
- **Middleware:** global route protection, role enforcement.

---

## 🧩 Key Directories
```
app/
 ├─ api/                 → REST endpoints
 │   ├─ auth/[...nextauth]/route.ts  → login/session
 │   ├─ links/route.ts              → GET/POST shortlinks
 │   └─ links/[slug]/route.ts       → PATCH/DELETE shortlink
 ├─ dashboard/           → overview page
 ├─ shortlinks/          → CRUD UI for shortlinks
 ├─ signin/              → login page + client component
 └─ layout.tsx           → main shell
components/              → UI pieces (Sidebar, Modal, etc.)
lib/                     → helpers (auth.ts, tools.ts)
prisma/                  → schema + seed
public/tools/            → static HTML tools
```

---

## 🪄 Adding / Modifying Features

### ➕ Add new roles
1. Create new user with desired `role` string (ex: `"VIEWER"`).
2. Update `middleware.ts` to restrict routes or methods:
   ```ts
   if (pathname.startsWith("/api/admin") && role !== "ADMIN") return deny();
   ```
3. Update client UI: hide buttons if `session.user.role !== 'ADMIN'`.

### 🧍 Add user management panel
Create `/admin/users/page.tsx`:
- List users (`GET /api/users`)
- Add/edit/delete (`POST/PATCH/DELETE /api/users`)
- Protect with `role === 'ADMIN'`.

### 🧱 Add new internal tool
1. Place your HTML file inside `/public/tools/<name>/index.html`.
2. Register it in `lib/tools.ts`:
   ```ts
   export const tools = [
     { name: "SERP Checker", path: "/tools/serp-checker" },
     { name: "Proxy Tester", path: "/tools/proxy-tester" },
     { name: "My Tool", path: "/tools/my-tool" }
   ];
   ```

### 🔗 Shortlink schema extension
`prisma/schema.prisma`
```prisma
model Link {
  id         String   @id @default(cuid())
  slug       String   @unique
  target     String
  clicks     Int      @default(0)
  expireAt   DateTime?
  oneTime    Boolean  @default(false)
  password   String?
  ownerId    String?
  createdAt  DateTime @default(now())
}
```

You can safely add:
- `category String?`
- `utm Json?`
Run:
```bash
npx prisma migrate dev -n add-category
```

---

## 🎨 Theme Customization
Edit `app/globals.css`:
```css
:root {
  --bg: #0f0a1f;
  --panel: #17122b;
  --panel-2: #1c1536;
  --text-muted: #9b8ec1;
  --hairline: rgba(124, 58, 237, .25);
  --grad-from: #e879f9;
  --grad-mid:  #8b5cf6;
  --grad-to:   #6366f1;
}
```
Utilities: `.card`, `.input`, `.btn-gradient`, `.btn-ghost`, `.btn-icon`, `.table`, `.chip`.

---

## 🧠 Notes for AI Agents
When modifying:
- Use **Next.js App Router conventions** (e.g. `route.ts` for API routes).
- Always update **middleware.ts** for access control.
- Respect `role` logic: `ADMIN` full, `OPERATOR` partial.
- Keep UI consistent with `globals.css` tokens.
- Avoid `bg-white`; use `.card` / `var(--panel)` instead.
- Prisma migration required for new DB fields.

---

## 🧩 Future Expansion Ideas
- User CRUD page with role editor.
- Link analytics (click count chart).
- Audit log of actions.
- Two-factor authentication (NextAuth adapter).
- API keys for automation.
- Multiple theme modes (`data-theme="midnight" | "violet"`).

---

## 🧪 Local Dev Quickstart
```bash
npm i
npx prisma migrate dev -n init
npm run seed
npm run dev
```
Login:  
`admin@example.com / admin123`

---

## 🧱 Deployment Tips
- For Vercel: switch to Postgres.
- For VPS/Docker: `npx prisma migrate deploy && npm run seed`.
- Use `.env` to override initial credentials.

---

**Created by:** kontol 🧑‍💻  
**Goal:** easy-to-extend private admin panel for internal tools & shortlinks.

## 🎨 UI Consistency Prompt (for AI Agents)
Ketika memperbaiki tampilan:
- Gunakan warna dari `globals.css` (`--bg`, `--panel`, `--muted`, `--line`, dll).
- Hindari `bg-white` atau `text-black`.
- Semua card/form pakai `.card` class (rounded, border halus, background `var(--panel)`).
- Input dan tombol pakai komponen custom (`.input`, `.btn-gradient`, `.btn-ghost`).
- Padding antar section harus simetris (`p-6` atau `p-8`).
- Font warna utama: `#EDEAFF`, font sekunder: `var(--muted)`.
- Tabel header pakai `background: #1c1536;`.
- Pastikan contrast tinggi tapi tetap lembut (dark mode style).

Contoh prompt:
> “Tolong rapikan form Shortlink biar konsisten dengan tema dark panel.  
> Semua input dan tombol seragam, tidak ada background putih.  
> Gunakan border lembut dan warna ungu dari palette utama.”
